<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\UserPoints;

class RetroactivePointsGrant extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'users:grant-retroactive-points {--dry-run : Show what would be done without actually doing it}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Grant points to existing users for already completed profile fields';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $isDryRun = $this->option('dry-run');
        
        if ($isDryRun) {
            $this->info('DRY RUN MODE: No changes will be made');
        }
        
        $this->info('Starting retroactive points grant for existing users...');
        
        $users = User::all();
        $totalPointsGranted = 0;
        $usersUpdated = 0;
        
        $progressBar = $this->output->createProgressBar($users->count());
        $progressBar->start();
        
        foreach ($users as $user) {
            $pointsGranted = 0;
            $actions = [];
            
            // Проверяем аватар
            if (!empty($user->avatar) && !$this->hasPointsForAction($user->id, 'avatar_upload')) {
                $pointsGranted += 25;
                $actions[] = 'avatar_upload';
                if (!$isDryRun) {
                    UserPoints::addPoints(
                        $user->id,
                        25,
                        'avatar_upload',
                        'Загрузка аватара (ретроактивно)',
                        $user
                    );
                }
            }
            
            // Проверяем имя
            if (!empty($user->name) && !$this->hasPointsForAction($user->id, 'profile_name')) {
                $pointsGranted += 10;
                $actions[] = 'profile_name';
                if (!$isDryRun) {
                    UserPoints::addPoints(
                        $user->id,
                        10,
                        'profile_name',
                        'Указание имени (ретроактивно)',
                        $user
                    );
                }
            }
            
            // Проверяем био
            if (!empty($user->bio) && !$this->hasPointsForAction($user->id, 'profile_bio')) {
                $pointsGranted += 15;
                $actions[] = 'profile_bio';
                if (!$isDryRun) {
                    UserPoints::addPoints(
                        $user->id,
                        15,
                        'profile_bio',
                        'Добавление биографии (ретроактивно)',
                        $user
                    );
                }
            }
            
            // Проверяем телефон
            if (!empty($user->phone) && !$this->hasPointsForAction($user->id, 'profile_phone')) {
                $pointsGranted += 10;
                $actions[] = 'profile_phone';
                if (!$isDryRun) {
                    UserPoints::addPoints(
                        $user->id,
                        10,
                        'profile_phone',
                        'Указание телефона (ретроактивно)',
                        $user
                    );
                }
            }
            
            // Проверяем дату рождения
            if (!empty($user->birth_date) && !$this->hasPointsForAction($user->id, 'profile_birth_date')) {
                $pointsGranted += 10;
                $actions[] = 'profile_birth_date';
                if (!$isDryRun) {
                    UserPoints::addPoints(
                        $user->id,
                        10,
                        'profile_birth_date',
                        'Указание даты рождения (ретроактивно)',
                        $user
                    );
                }
            }
            
            // Проверяем пол
            if (!empty($user->gender) && !$this->hasPointsForAction($user->id, 'profile_gender')) {
                $pointsGranted += 5;
                $actions[] = 'profile_gender';
                if (!$isDryRun) {
                    UserPoints::addPoints(
                        $user->id,
                        5,
                        'profile_gender',
                        'Указание пола (ретроактивно)',
                        $user
                    );
                }
            }
            
            // Проверяем страну
            if (!empty($user->country) && !$this->hasPointsForAction($user->id, 'profile_country')) {
                $pointsGranted += 8;
                $actions[] = 'profile_country';
                if (!$isDryRun) {
                    UserPoints::addPoints(
                        $user->id,
                        8,
                        'profile_country',
                        'Указание страны (ретроактивно)',
                        $user
                    );
                }
            }
            
            // Проверяем город
            if (!empty($user->city) && !$this->hasPointsForAction($user->id, 'profile_city')) {
                $pointsGranted += 8;
                $actions[] = 'profile_city';
                if (!$isDryRun) {
                    UserPoints::addPoints(
                        $user->id,
                        8,
                        'profile_city',
                        'Указание города (ретроактивно)',
                        $user
                    );
                }
            }
            
            // Проверяем полный профиль (все поля + аватар)
            if ($this->isProfileComplete($user) && !$this->hasPointsForAction($user->id, 'profile_complete')) {
                $pointsGranted += 50;
                $actions[] = 'profile_complete';
                if (!$isDryRun) {
                    UserPoints::addPoints(
                        $user->id,
                        50,
                        'profile_complete',
                        'Полное заполнение профиля (ретроактивно)',
                        $user
                    );
                }
            }
            
            if ($pointsGranted > 0) {
                $totalPointsGranted += $pointsGranted;
                $usersUpdated++;
                
                if (!$isDryRun) {
                    // Обновляем ранг пользователя
                    $user->updateRank();
                }
            }
            
            $progressBar->advance();
        }
        
        $progressBar->finish();
        
        $this->newLine();
        if ($isDryRun) {
            $this->info("DRY RUN RESULTS:");
        } else {
            $this->info("Retroactive points grant completed!");
        }
        $this->info("Total users processed: {$users->count()}");
        $this->info("Users that would receive points: {$usersUpdated}");
        $this->info("Total points that would be granted: {$totalPointsGranted}");
        
        return 0;
    }
    
    /**
     * Проверяет, есть ли уже очки за определенное действие у пользователя
     */
    private function hasPointsForAction(int $userId, string $actionType): bool
    {
        return UserPoints::where('user_id', $userId)
            ->where('action_type', $actionType)
            ->exists();
    }
    
    /**
     * Проверяет, полностью ли заполнен профиль пользователя
     */
    private function isProfileComplete(User $user): bool
    {
        return !empty($user->avatar) &&
               !empty($user->name) &&
               !empty($user->bio) &&
               !empty($user->phone) &&
               !empty($user->birth_date) &&
               !empty($user->gender) &&
               !empty($user->country) &&
               !empty($user->city);
    }
} 