<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Symfony\Component\Process\Process;

class ServeSimple extends Command
{
    protected $signature = 'serve:simple {--port=8000} {--vite}';
    protected $description = 'Запуск Laravel + Vite серверов';

    public function handle()
    {
        $port = $this->option('port');
        $vite = $this->option('vite');

        $this->info('🚀 Запуск серверов...');
        $this->info("📡 Laravel: http://localhost:{$port}");
        
        if ($vite) {
            $this->info('⚡ Vite: http://localhost:5173');
        }
        
        $this->info('');
        $this->info('💡 Нажмите Ctrl+C для остановки');
        $this->info('');

        if ($vite) {
            // Запускаем оба сервера параллельно
            $this->startBothServers($port);
        } else {
            // Запускаем только Laravel
            $this->startLaravelOnly($port);
        }

        return 0;
    }

    private function startBothServers($port)
    {
        $this->info('🔄 Запуск Laravel и Vite серверов...');
        
        // Создаем процессы
        $laravelProcess = new Process(['php', 'artisan', 'serve', '--port=' . $port]);
        $viteProcess = new Process(['npm', 'run', 'dev']);
        
        // Настраиваем процессы
        $laravelProcess->setTimeout(null);
        $viteProcess->setTimeout(null);
        
        // Запускаем Laravel
        $laravelProcess->start();
        $this->info('✅ Laravel сервер запущен');
        
        // Ждем немного
        sleep(2);
        
        // Запускаем Vite
        $viteProcess->start();
        $this->info('✅ Vite сервер запущен');
        
        $this->info('');
        $this->info('🎉 Оба сервера запущены!');
        $this->info('🌐 Откройте: http://localhost:' . $port);
        $this->info('');
        
        // Ждем завершения процессов
        while ($laravelProcess->isRunning() || $viteProcess->isRunning()) {
            sleep(1);
            
            // Проверяем вывод
            if ($laravelProcess->isRunning()) {
                $output = $laravelProcess->getIncrementalOutput();
                if ($output) {
                    $this->line($output);
                }
            }
            
            if ($viteProcess->isRunning()) {
                $output = $viteProcess->getIncrementalOutput();
                if ($output) {
                    $this->line($output);
                }
            }
        }
        
        // Останавливаем процессы
        if ($laravelProcess->isRunning()) {
            $laravelProcess->stop();
        }
        if ($viteProcess->isRunning()) {
            $viteProcess->stop();
        }
    }

    private function startLaravelOnly($port)
    {
        $this->info('🔄 Запуск только Laravel сервера...');
        
        $laravelProcess = new Process(['php', 'artisan', 'serve', '--port=' . $port]);
        $laravelProcess->setTty(true);
        $laravelProcess->run();
    }
} 