<?php

namespace App\Console\Commands;

use App\Models\ParserJob;
use App\Services\ParserService;
use Illuminate\Console\Command;

class TestParser extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'parser:test {url} {--selectors=}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Test parser selectors on a given URL';

    /**
     * Execute the console command.
     */
    public function handle(ParserService $parserService)
    {
        $url = $this->argument('url');
        $selectors = $this->option('selectors');

        if (!$selectors) {
            $this->error('Please provide selectors using --selectors option');
            return 1;
        }

        $selectorsArray = json_decode($selectors, true);
        if (!$selectorsArray) {
            $this->error('Invalid JSON format for selectors');
            return 1;
        }

        $this->info("Testing selectors on: {$url}");
        $this->line('');

        try {
            $results = $parserService->testSelectors($url, $selectorsArray);

            if (isset($results['error'])) {
                $this->error("Error: {$results['error']}");
                return 1;
            }

            foreach ($results as $selector => $result) {
                $this->line("Selector: {$selector}");
                
                if (isset($result['error'])) {
                    $this->error("  Error: {$result['error']}");
                } else {
                    $this->info("  Found: {$result['count']} elements");
                    if ($result['sample']) {
                        $this->line("  Sample: {$result['sample']}");
                    }
                }
                $this->line('');
            }

        } catch (\Exception $e) {
            $this->error("Error: {$e->getMessage()}");
            return 1;
        }

        return 0;
    }
}
