<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Bookmaker;
use App\Models\Casino;
use App\Models\PromoCode;
use App\Models\Prediction;
use App\Models\Category;
use App\Models\UserPoints;
use App\Models\PromoCodeUsage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class AdminController extends Controller
{
    /**
     * Get dashboard statistics
     */
    public function dashboard()
    {
        $stats = [
            'total_users' => User::count(),
            'new_users_today' => User::whereDate('created_at', today())->count(),
            'total_bookmakers' => Bookmaker::count(),
            'active_bookmakers' => Bookmaker::where('is_active', true)->count(),
            'total_casinos' => Casino::count(),
            'active_casinos' => Casino::where('is_active', true)->count(),
            'total_games' => Casino::sum('games_count'),
            'total_promo_codes' => PromoCode::count(),
            'active_promo_codes' => PromoCode::where('is_active', true)->count(),
            'total_predictions' => Prediction::count(),
            'pending_predictions' => Prediction::where('status', 'pending')->count(),
            'total_points_earned' => UserPoints::sum('points'),
            'total_promo_uses' => PromoCodeUsage::count(),
        ];

        // Recent activity
        $recentUsers = User::latest()->limit(5)->get(['id', 'name', 'email', 'created_at']);
        $recentPredictions = Prediction::with('user:id,name,username')->latest()->limit(5)->get();
        $recentPromoUses = PromoCodeUsage::with(['user:id,name', 'promoCode:id,title'])->latest()->limit(5)->get();

        return response()->json([
            'success' => true,
            'data' => [
                'statistics' => $stats,
                'recent_users' => $recentUsers,
                'recent_predictions' => $recentPredictions,
                'recent_promo_uses' => $recentPromoUses
            ]
        ]);
    }

    /**
     * Get all users with pagination and filters
     */
    public function users(Request $request)
    {
        $query = User::with(['referrer:id,name,username']);

        // Apply filters
        if ($request->has('role')) {
            $query->where('role', $request->role);
        }

        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('username', 'like', "%{$search}%");
            });
        }

        if ($request->has('verified')) {
            $query->where('is_verified', $request->verified);
        }

        $users = $query->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 15));

        return response()->json([
            'success' => true,
            'data' => [
                'users' => $users
            ]
        ]);
    }

    /**
     * Update user role
     */
    public function updateUserRole(Request $request, $userId)
    {
        $validator = Validator::make($request->all(), [
            'role' => 'required|in:user,admin,moderator'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::findOrFail($userId);
        $user->update(['role' => $request->role]);

        return response()->json([
            'success' => true,
            'message' => 'User role updated successfully',
            'data' => [
                'user' => $user
            ]
        ]);
    }

    /**
     * Toggle user status
     */
    public function toggleUserStatus($userId)
    {
        $user = User::findOrFail($userId);
        
        // Отладочная информация
        \Log::info("Toggle user status - User ID: {$userId}, Current is_verified: " . ($user->is_verified ? '1' : '0'));
        
        // Явно переключаем статус is_verified
        $newStatus = $user->is_verified ? 0 : 1;
        $user->is_verified = $newStatus;
        $user->save();
        
        \Log::info("Toggle user status - New is_verified: " . $newStatus);

        return response()->json([
            'success' => true,
            'message' => 'User status updated successfully',
            'data' => [
                'user' => $user->fresh()
            ]
        ]);
    }

    /**
     * Get all bookmakers with management options
     */
    public function bookmakers(Request $request)
    {
        $query = Bookmaker::with('category');

        if ($request->has('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        if ($request->has('active')) {
            $query->where('is_active', $request->active);
        }

        if ($request->has('featured')) {
            $query->where('is_featured', $request->featured);
        }

        $bookmakers = $query->orderBy('sort_order')->paginate($request->get('per_page', 15));

        return response()->json([
            'success' => true,
            'data' => [
                'bookmakers' => $bookmakers
            ]
        ]);
    }

    /**
     * Store new bookmaker
     */
    public function storeBookmaker(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'logo' => 'nullable|string',
            'website_url' => 'nullable|url',
            'category_id' => 'nullable|exists:categories,id',
            'is_active' => 'boolean',
            'is_featured' => 'boolean',
            'sort_order' => 'integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $bookmaker = Bookmaker::create($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Bookmaker created successfully',
            'data' => [
                'bookmaker' => $bookmaker
            ]
        ]);
    }

    /**
     * Update bookmaker
     */
    public function updateBookmaker(Request $request, $bookmakerId)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'logo' => 'nullable|string',
            'website_url' => 'nullable|url',
            'category_id' => 'nullable|exists:categories,id',
            'is_active' => 'boolean',
            'is_featured' => 'boolean',
            'sort_order' => 'integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $bookmaker = Bookmaker::findOrFail($bookmakerId);
        $bookmaker->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Bookmaker updated successfully',
            'data' => [
                'bookmaker' => $bookmaker
            ]
        ]);
    }

    /**
     * Toggle bookmaker status
     */
    public function toggleBookmakerStatus($bookmakerId)
    {
        $bookmaker = Bookmaker::findOrFail($bookmakerId);
        $bookmaker->update(['is_active' => !$bookmaker->is_active]);

        return response()->json([
            'success' => true,
            'message' => 'Bookmaker status updated successfully',
            'data' => [
                'bookmaker' => $bookmaker
            ]
        ]);
    }

    /**
     * Get all casinos with management options
     */
    public function casinos(Request $request)
    {
        $query = Casino::query();

        if ($request->has('active')) {
            $query->where('is_active', $request->active);
        }

        if ($request->has('featured')) {
            $query->where('is_featured', $request->featured);
        }

        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        $casinos = $query->orderBy('created_at', 'desc')->paginate($request->get('per_page', 15));

        return response()->json([
            'success' => true,
            'data' => [
                'casinos' => $casinos
            ]
        ]);
    }

    /**
     * Store new casino
     */
    public function storeCasino(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:casinos',
            'description' => 'nullable|string',
            'logo' => 'nullable|string',
            'website' => 'nullable|url',
            'rating' => 'nullable|numeric|min:0|max:5',
            'min_deposit' => 'nullable|numeric|min:0',
            'max_deposit' => 'nullable|numeric|min:0',
            'min_withdrawal' => 'nullable|numeric|min:0',
            'max_withdrawal' => 'nullable|numeric|min:0',
            'withdrawal_time' => 'nullable|string|max:255',
            'license' => 'nullable|string|max:255',
            'founded_year' => 'nullable|integer|min:1900|max:' . date('Y'),
            'headquarters' => 'nullable|string|max:255',
            'languages' => 'nullable|array',
            'currencies' => 'nullable|array',
            'payment_methods' => 'nullable|array',
            'games_count' => 'nullable|integer|min:0',
            'providers_count' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
            'is_featured' => 'boolean',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'meta_keywords' => 'nullable|string|max:500'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $casino = Casino::create($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Casino created successfully',
            'data' => [
                'casino' => $casino
            ]
        ]);
    }

    /**
     * Update casino
     */
    public function updateCasino(Request $request, $casinoId)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:casinos,slug,' . $casinoId,
            'description' => 'nullable|string',
            'logo' => 'nullable|string',
            'website' => 'nullable|url',
            'rating' => 'nullable|numeric|min:0|max:5',
            'min_deposit' => 'nullable|numeric|min:0',
            'max_deposit' => 'nullable|numeric|min:0',
            'min_withdrawal' => 'nullable|numeric|min:0',
            'max_withdrawal' => 'nullable|numeric|min:0',
            'withdrawal_time' => 'nullable|string|max:255',
            'license' => 'nullable|string|max:255',
            'founded_year' => 'nullable|integer|min:1900|max:' . date('Y'),
            'headquarters' => 'nullable|string|max:255',
            'languages' => 'nullable|array',
            'currencies' => 'nullable|array',
            'payment_methods' => 'nullable|array',
            'games_count' => 'nullable|integer|min:0',
            'providers_count' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
            'is_featured' => 'boolean',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'meta_keywords' => 'nullable|string|max:500'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $casino = Casino::findOrFail($casinoId);
        $casino->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Casino updated successfully',
            'data' => [
                'casino' => $casino
            ]
        ]);
    }

    /**
     * Toggle casino status
     */
    public function toggleCasinoStatus($casinoId)
    {
        $casino = Casino::findOrFail($casinoId);
        $casino->update(['is_active' => !$casino->is_active]);

        return response()->json([
            'success' => true,
            'message' => 'Casino status updated successfully',
            'data' => [
                'casino' => $casino
            ]
        ]);
    }

    /**
     * Upload casino logo
     */
    public function uploadCasinoLogo(Request $request, $casinoId)
    {
        $validator = Validator::make($request->all(), [
            'logo' => 'required|image|mimes:jpeg,png,jpg,gif,svg,webp|max:2048'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $casino = Casino::findOrFail($casinoId);

        if ($request->hasFile('logo')) {
            // Удаляем старый логотип, если он существует
            if ($casino->logo && file_exists(storage_path('app/public/' . $casino->logo))) {
                unlink(storage_path('app/public/' . $casino->logo));
            }

            // Сохраняем новый логотип
            $logoPath = $request->file('logo')->store('casinos/logos', 'public');
            
            // Обновляем казино
            $casino->update(['logo' => $logoPath]);

            return response()->json([
                'success' => true,
                'message' => 'Casino logo uploaded successfully',
                'data' => [
                    'casino' => $casino,
                    'logo_url' => asset('storage/' . $logoPath)
                ]
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'No logo file provided'
        ], 400);
    }

    /**
     * Delete casino
     */
    public function deleteCasino($casinoId)
    {
        $casino = Casino::findOrFail($casinoId);
        
        // Удаляем логотип, если он существует
        if ($casino->logo && file_exists(storage_path('app/public/' . $casino->logo))) {
            unlink(storage_path('app/public/' . $casino->logo));
        }
        
        $casino->delete();

        return response()->json([
            'success' => true,
            'message' => 'Casino deleted successfully'
        ]);
    }

    /**
     * Duplicate casino
     */
    public function duplicateCasino($casinoId)
    {
        $originalCasino = Casino::findOrFail($casinoId);
        
        // Создаем копию казино
        $newCasino = $originalCasino->replicate();
        $newCasino->name = $originalCasino->name . ' (Копия)';
        $newCasino->slug = $originalCasino->slug . '-copy-' . time();
        $newCasino->is_active = false; // Копия по умолчанию неактивна
        $newCasino->is_featured = false; // Копия по умолчанию не в избранном
        $newCasino->save();
        
        // Копируем логотип, если он есть
        if ($originalCasino->logo && file_exists(storage_path('app/public/' . $originalCasino->logo))) {
            $originalPath = storage_path('app/public/' . $originalCasino->logo);
            $extension = pathinfo($originalPath, PATHINFO_EXTENSION);
            $newFileName = 'casinos/logos/' . uniqid() . '.' . $extension;
            $newPath = storage_path('app/public/' . $newFileName);
            
            // Создаем директорию, если её нет
            $directory = dirname($newPath);
            if (!file_exists($directory)) {
                mkdir($directory, 0755, true);
            }
            
            // Копируем файл
            copy($originalPath, $newPath);
            $newCasino->update(['logo' => $newFileName]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Casino duplicated successfully',
            'data' => [
                'casino' => $newCasino
            ]
        ]);
    }

    /**
     * Get all promo codes with management options
     */
    public function promoCodes(Request $request)
    {
        $query = PromoCode::with('bookmaker');

        if ($request->has('bookmaker_id')) {
            $query->where('bookmaker_id', $request->bookmaker_id);
        }

        if ($request->has('active')) {
            $query->where('is_active', $request->active);
        }

        if ($request->has('featured')) {
            $query->where('is_featured', $request->featured);
        }

        $promoCodes = $query->orderBy('created_at', 'desc')->paginate($request->get('per_page', 15));

        return response()->json([
            'success' => true,
            'data' => [
                'promo_codes' => $promoCodes
            ]
        ]);
    }

    /**
     * Store new promo code
     */
    public function storePromoCode(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'code' => 'required|string|max:255',
            'bookmaker_id' => 'required|exists:bookmakers,id',
            'is_active' => 'boolean',
            'is_featured' => 'boolean',
            'sort_order' => 'integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $promoCode = PromoCode::create($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Promo code created successfully',
            'data' => [
                'promo_code' => $promoCode
            ]
        ]);
    }

    /**
     * Update promo code
     */
    public function updatePromoCode(Request $request, $promoCodeId)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'code' => 'required|string|max:255',
            'bookmaker_id' => 'required|exists:bookmakers,id',
            'is_active' => 'boolean',
            'is_featured' => 'boolean',
            'sort_order' => 'integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $promoCode = PromoCode::findOrFail($promoCodeId);
        $promoCode->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Promo code updated successfully',
            'data' => [
                'promo_code' => $promoCode
            ]
        ]);
    }

    /**
     * Toggle promo code status
     */
    public function togglePromoCodeStatus($promoCodeId)
    {
        $promoCode = PromoCode::findOrFail($promoCodeId);
        $promoCode->update(['is_active' => !$promoCode->is_active]);

        return response()->json([
            'success' => true,
            'message' => 'Promo code status updated successfully',
            'data' => [
                'promo_code' => $promoCode
            ]
        ]);
    }

    /**
     * Get all predictions with management options
     */
    public function predictions(Request $request)
    {
        $query = Prediction::with('user:id,name,username');

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('sport')) {
            $query->where('sport', $request->sport);
        }

        if ($request->has('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        $predictions = $query->orderBy('match_date', 'desc')->paginate($request->get('per_page', 15));

        return response()->json([
            'success' => true,
            'data' => [
                'predictions' => $predictions
            ]
        ]);
    }

    /**
     * Update prediction status
     */
    public function updatePredictionStatus(Request $request, $predictionId)
    {
        $validator = Validator::make($request->all(), [
            'status' => 'required|in:pending,won,lost,void',
            'profit_loss' => 'nullable|numeric'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $prediction = Prediction::findOrFail($predictionId);
        $prediction->update($request->only(['status', 'profit_loss']));

        // Update user statistics
        $prediction->updateUserStats();

        return response()->json([
            'success' => true,
            'message' => 'Prediction status updated successfully',
            'data' => [
                'prediction' => $prediction->load('user')
            ]
        ]);
    }

    /**
     * Get public statistics (для всех пользователей)
     */
    public function publicStats()
    {
        $stats = [
            'total_users' => User::count(),
            'total_bookmakers' => Bookmaker::where('is_active', true)->count(),
            'total_promo_codes' => PromoCode::where('is_active', true)->count(),
            'total_predictions' => Prediction::count(),
            'total_points_earned' => UserPoints::sum('points'),
        ];

        return response()->json([
            'success' => true,
            'data' => $stats
        ]);
    }

    /**
     * Get system statistics (только для админов)
     */
    public function systemStats()
    {
        $stats = [
            'totalUsers' => User::count(),
            'newUsersThisMonth' => User::whereMonth('created_at', now()->month)->count(),
            'totalBookmakers' => Bookmaker::count(),
            'activeBookmakers' => Bookmaker::where('is_active', true)->count(),
            'totalCasinos' => Casino::count(),
            'activeCasinos' => Casino::where('is_active', true)->count(),
            'totalGames' => Casino::sum('games_count'),
            'totalPromoCodes' => PromoCode::count(),
            'activePromoCodes' => PromoCode::where('is_active', true)->count(),
            'totalUsages' => PromoCodeUsage::count(),
            'usagesThisMonth' => PromoCodeUsage::whereMonth('created_at', now()->month)->count(),
        ];

        // Recent activity
        $recentUsers = User::latest()->limit(5)->get(['id', 'name', 'email', 'username', 'created_at']);
        $recentUsages = PromoCodeUsage::with(['user:id,name,username', 'promoCode:id,title,code'])
            ->latest()
            ->limit(5)
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'statistics' => $stats,
                'recent_users' => $recentUsers,
                'recent_promo_uses' => $recentUsages
            ]
        ]);
    }
}
