<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Illuminate\Validation\Rules\Password as PasswordRules;
use App\Mail\ResetPassword;

class AuthController extends Controller
{
    /**
     * Register a new user
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => ['required', 'confirmed', PasswordRules::defaults()],
            'username' => 'required|string|max:255|unique:users',
            'referral_code' => 'nullable|string|exists:users,referral_code',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'username' => $request->username,
            'referral_code' => User::generateReferralCode(),
            'referred_by' => $request->referral_code ? User::where('referral_code', $request->referral_code)->first()?->id : null,
        ]);

        // Add referral bonus if user was referred
        if ($user->referred_by) {
            $user->increment('total_points', 100);
            $user->increment('available_points', 100);
        }

        $token = $user->createToken('auth_token')->plainTextToken;

        // Подготавливаем данные пользователя с avatar_url
        $userData = $user->toArray();
        $userData['avatar_url'] = $user->getAvatarUrl();

        return response()->json([
            'success' => true,
            'message' => 'User registered successfully',
            'data' => [
                'user' => $userData,
                'token' => $token,
                'token_type' => 'Bearer'
            ]
        ], 201);
    }

    /**
     * Login user
     */
    public function login(Request $request)
    {
        \Log::info('🔑 Login attempt', [
            'email' => $request->email,
            'headers' => $request->header(),
            'ip' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'timestamp' => now()->toISOString()
        ]);

        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email',
            'password' => 'required|string',
        ]);

        if ($validator->fails()) {
            \Log::warning('Login validation failed', [
                'errors' => $validator->errors()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        if (!Auth::attempt($request->only('email', 'password'))) {
            \Log::warning('Login failed - invalid credentials', [
                'email' => $request->email
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Invalid credentials'
            ], 401);
        }

        $user = User::where('email', $request->email)->firstOrFail();
        $user->update(['last_login_at' => now()]);

        $token = $user->createToken('auth_token')->plainTextToken;

        \Log::info('✅ Login successful', [
            'user_id' => $user->id,
            'user_email' => $user->email,
            'token_created' => true,
            'token_preview' => substr($token, 0, 20) . '...',
            'timestamp' => now()->toISOString()
        ]);

        $userData = $user->toArray();
        $userData['avatar_url'] = $user->getAvatarUrl();

        return response()->json([
            'success' => true,
            'message' => 'Login successful',
            'data' => [
                'user' => $userData,
                'token' => $token,
                'token_type' => 'Bearer'
            ]
        ]);
    }

    /**
     * Logout user
     */
    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'success' => true,
            'message' => 'Logged out successfully'
        ]);
    }

    /**
     * Get authenticated user profile
     */
    public function profile(Request $request)
    {
        $user = $request->user()->load(['referrals', 'predictions', 'pointsHistory']);

        // Подготавливаем данные пользователя с avatar_url
        $userData = $user->toArray();
        $userData['avatar_url'] = $user->getAvatarUrl();

        return response()->json([
            'success' => true,
            'data' => [
                'user' => $userData
            ]
        ]);
    }

    /**
     * Update user profile
     */
    public function updateProfile(Request $request)
    {
        $user = $request->user();

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|string|max:255',
            'username' => 'sometimes|string|max:255|unique:users,username,' . $user->id,
            'bio' => 'sometimes|string|max:1000',
            'phone' => 'sometimes|string|max:20',
            'birth_date' => 'sometimes|date',
            'country' => 'sometimes|string|max:100',
            'city' => 'sometimes|string|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $user->update($request->only([
            'name',
            'username',
            'bio',
            'phone',
            'birth_date',
            'country',
            'city'
        ]));

        // Обновляем пользователя для получения свежих данных
        $user = $user->fresh();

        // Подготавливаем данные пользователя с avatar_url
        $userData = $user->toArray();
        $userData['avatar_url'] = $user->getAvatarUrl();

        return response()->json([
            'success' => true,
            'message' => 'Profile updated successfully',
            'data' => [
                'user' => $userData
            ]
        ]);
    }

    /**
     * Send password reset link
     */
    public function forgotPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email|exists:users,email',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::where('email', $request->email)->first();

        // Генерируем токен
        $token = Str::random(64);

        // Сохраняем токен в базе
        \DB::table('password_reset_tokens')->updateOrInsert(
            ['email' => $request->email],
            [
                'email' => $request->email,
                'token' => $token,
                'created_at' => now()
            ]
        );

        try {
            Mail::to($user->email)->send(new ResetPassword($token, $user));

            \Log::info('Reset password link: ' . url('/reset-password?token=' . $token . '&email=' . $user->email));

            return response()->json([
                'success' => true,
                'message' => 'Ссылка для восстановления пароля отправлена на ваш email'
            ]);
        } catch (\Exception $e) {
            \Log::info('Reset password link (email failed): ' . url('/reset-password?token=' . $token . '&email=' . $user->email));

            return response()->json([
                'success' => true,
                'message' => 'Ссылка для восстановления пароля отправлена на ваш email'
            ]);
        }
    }

    /**
     * Reset password
     */
    public function resetPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'token' => 'required|string',
            'email' => 'required|email|exists:users,email',
            'password' => ['required', 'confirmed', PasswordRules::defaults()],
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        // Проверяем токен
        $resetToken = \DB::table('password_reset_tokens')
            ->where('email', $request->email)
            ->where('token', $request->token)
            ->first();

        if (!$resetToken) {
            return response()->json([
                'success' => false,
                'message' => 'Недействительный токен восстановления'
            ], 400);
        }

        // Проверяем время жизни токена (60 минут)
        if (now()->diffInMinutes($resetToken->created_at) > 60) {
            \DB::table('password_reset_tokens')->where('email', $request->email)->delete();
            return response()->json([
                'success' => false,
                'message' => 'Срок действия токена истек. Запросите новую ссылку.'
            ], 400);
        }

        // Обновляем пароль
        $user = User::where('email', $request->email)->first();
        $user->update([
            'password' => Hash::make($request->password)
        ]);

        // Удаляем использованный токен
        \DB::table('password_reset_tokens')->where('email', $request->email)->delete();

        return response()->json([
            'success' => true,
            'message' => 'Пароль успешно изменен'
        ]);
    }

    /**
     * Verify reset token
     */
    public function verifyResetToken(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'token' => 'required|string',
            'email' => 'required|email|exists:users,email',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $resetToken = \DB::table('password_reset_tokens')
            ->where('email', $request->email)
            ->where('token', $request->token)
            ->first();

        if (!$resetToken) {
            return response()->json([
                'success' => false,
                'message' => 'Недействительный токен восстановления'
            ], 400);
        }

        // Проверяем время жизни токена
        if (now()->diffInMinutes($resetToken->created_at) > 60) {
            \DB::table('password_reset_tokens')->where('email', $request->email)->delete();
            return response()->json([
                'success' => false,
                'message' => 'Срок действия токена истек'
            ], 400);
        }

        return response()->json([
            'success' => true,
            'message' => 'Токен действителен'
        ]);
    }
}
