<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\UserBlacklist;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class BlacklistController extends Controller
{
    /**
     * Получить список заблокированных пользователей
     */
    public function index(Request $request): JsonResponse
    {
        $user = Auth::user();
        
        $blockedUsers = UserBlacklist::where('user_id', $user->id)
            ->with(['blockedUser' => function ($query) {
                $query->select('id', 'username', 'name', 'avatar', 'bio', 'is_online', 'last_activity_at');
            }])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return response()->json([
            'success' => true,
            'data' => [
                'blocked_users' => $blockedUsers->items(),
                'pagination' => [
                    'current_page' => $blockedUsers->currentPage(),
                    'last_page' => $blockedUsers->lastPage(),
                    'per_page' => $blockedUsers->perPage(),
                    'total' => $blockedUsers->total(),
                ]
            ]
        ]);
    }

    /**
     * Заблокировать пользователя
     */
    public function block(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|integer|exists:users,id',
            'reason' => 'nullable|string|max:500'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Ошибка валидации',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = Auth::user();
        $blockedUserId = $request->input('user_id');
        $reason = $request->input('reason');

        // Нельзя заблокировать самого себя
        if ($user->id === $blockedUserId) {
            return response()->json([
                'success' => false,
                'message' => 'Нельзя заблокировать самого себя'
            ], 400);
        }

        // Проверяем, не заблокирован ли уже
        if ($user->hasBlockedUser($blockedUserId)) {
            return response()->json([
                'success' => false,
                'message' => 'Пользователь уже заблокирован'
            ], 400);
        }

        // Заблокируем пользователя
        if ($user->blockUser($blockedUserId, $reason)) {
            return response()->json([
                'success' => true,
                'message' => 'Пользователь успешно заблокирован'
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Ошибка при блокировке пользователя'
        ], 500);
    }

    /**
     * Разблокировать пользователя
     */
    public function unblock(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|integer|exists:users,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Ошибка валидации',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = Auth::user();
        $blockedUserId = $request->input('user_id');

        // Проверяем, заблокирован ли пользователь
        if (!$user->hasBlockedUser($blockedUserId)) {
            return response()->json([
                'success' => false,
                'message' => 'Пользователь не заблокирован'
            ], 400);
        }

        // Разблокируем пользователя
        if ($user->unblockUser($blockedUserId)) {
            return response()->json([
                'success' => true,
                'message' => 'Пользователь успешно разблокирован'
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Ошибка при разблокировке пользователя'
        ], 500);
    }

    /**
     * Проверить, заблокирован ли пользователь
     */
    public function check(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|integer|exists:users,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Ошибка валидации',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = Auth::user();
        $checkUserId = $request->input('user_id');

        $isBlocked = $user->hasBlockedUser($checkUserId);
        $isBlockedBy = $user->isBlockedBy($checkUserId);

        return response()->json([
            'success' => true,
            'data' => [
                'is_blocked' => $isBlocked,
                'is_blocked_by' => $isBlockedBy,
                'can_interact' => !$isBlocked && !$isBlockedBy
            ]
        ]);
    }

    /**
     * Получить статистику черного списка
     */
    public function stats(): JsonResponse
    {
        $user = Auth::user();
        
        $blockedCount = UserBlacklist::where('user_id', $user->id)->count();
        $blockedByCount = UserBlacklist::where('blocked_user_id', $user->id)->count();

        return response()->json([
            'success' => true,
            'data' => [
                'blocked_count' => $blockedCount,
                'blocked_by_count' => $blockedByCount
            ]
        ]);
    }
}
