<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Casino;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CasinoController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Casino::query();

        // Search
        if ($request->search) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // Filter by rating
        if ($request->rating) {
            $query->where('rating', '>=', $request->rating);
        }

        // Filter by min deposit
        if ($request->min_deposit) {
            $query->where('min_deposit', '<=', $request->min_deposit);
        }

        // Filter by featured
        if ($request->featured) {
            $query->featured();
        }

        // Sort
        $sort = $request->sort ?? 'rating';
        $order = $request->order ?? 'desc';
        $query->orderBy($sort, $order);

        $casinos = $query->paginate($request->per_page ?? 12);

        return response()->json([
            'success' => true,
            'data' => [
                'casinos' => $casinos
            ]
        ]);
    }

    /**
     * Display the specified resource.
     */
    public function show($slug)
    {
        $casino = Casino::where('slug', $slug)->first();
        
        if (!$casino) {
            return response()->json([
                'success' => false,
                'message' => 'Casino not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'casino' => $casino
            ]
        ]);
    }

    /**
     * Search casinos
     */
    public function search(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'query' => 'required|string|min:2'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $query = $request->query;
        $casinos = Casino::query()
            ->where(function ($q) use ($query) {
                $q->where('name', 'like', "%{$query}%")
                  ->orWhere('description', 'like', "%{$query}%");
            })
            ->orderBy('rating', 'desc')
            ->limit(10)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $casinos
        ]);
    }

    /**
     * Get featured casinos
     */
    public function featured()
    {
        $casinos = Casino::query()
            ->featured()
            ->orderBy('rating', 'desc')
            ->limit(6)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $casinos
        ]);
    }

    /**
     * Get top rated casinos
     */
    public function topRated()
    {
        $casinos = Casino::query()
            ->topRated()
            ->limit(10)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $casinos
        ]);
    }

    /**
     * Get casino games
     */
    public function games(Casino $casino)
    {
        // Пока возвращаем пустой массив, так как у нас нет модели игр
        // В будущем можно создать модель CasinoGame и связать с Casino
        $games = [];

        return response()->json([
            'success' => true,
            'data' => [
                'games' => $games
            ]
        ]);
    }
} 