<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\News;
use App\Models\NewsComment;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class NewsCommentController extends Controller
{
    /**
     * Получить комментарии к новости (публичный метод)
     */
    public function index(Request $request, $newsId): JsonResponse
    {
        try {
            // Проверяем существование новости
            $news = News::find($newsId);
            if (!$news) {
                return response()->json([
                    'success' => false,
                    'message' => 'Новость не найдена'
                ], 404);
            }
            
            // Проверяем все комментарии к новости
            $allComments = NewsComment::where('news_id', $news->id)->get();
            \Log::info('All comments for news', [
                'news_id' => $news->id,
                'total_comments' => $allComments->count(),
                'comments_data' => $allComments->map(function($c) {
                    return [
                        'id' => $c->id,
                        'user_id' => $c->user_id,
                        'parent_id' => $c->parent_id,
                        'status' => $c->status,
                        'content' => substr($c->content, 0, 50) . '...'
                    ];
                })
            ]);
            
            // Загружаем комментарии (только корневые и одобренные) с ответами
            $comments = NewsComment::where('news_id', $news->id)
                ->whereNull('parent_id')
                ->where('status', 'approved')
                ->with([
                    'user' => function($query) {
                        $query->select('id', 'username', 'name', 'avatar');
                    },
                    'replies.user' => function($query) {
                        $query->select('id', 'username', 'name', 'avatar');
                    }
                ])
                ->orderBy('created_at', 'desc')
                ->get();

            \Log::info('Filtered comments', [
                'filtered_count' => $comments->count(),
                'comments_data' => $comments->map(function($c) {
                    return [
                        'id' => $c->id,
                        'user_id' => $c->user_id,
                        'parent_id' => $c->parent_id,
                        'status' => $c->status,
                        'content' => substr($c->content, 0, 50) . '...'
                    ];
                })
            ]);

            // Добавляем базовую информацию
            $isAuthenticated = auth('sanctum')->check();
            $currentUserId = $isAuthenticated ? auth('sanctum')->id() : null;
            
            \Log::info('News comments auth check', [
                'is_authenticated' => $isAuthenticated,
                'current_user_id' => $currentUserId,
                'comments_count' => $comments->count()
            ]);

            $comments->transform(function ($comment) use ($isAuthenticated, $currentUserId) {
                $comment->is_liked = false;
                $comment->can_reply = $isAuthenticated;
                $comment->can_edit = $isAuthenticated && $comment->user_id === $currentUserId;
                $comment->can_delete = $isAuthenticated && $comment->user_id === $currentUserId;
                
                // Добавляем avatar_url к пользователю (как в getPredictors)
                if ($comment->user) {
                    $comment->user->avatar_url = $comment->user->getAvatarUrl();
                    \Log::info('Comment user avatar', [
                        'user_id' => $comment->user->id,
                        'username' => $comment->user->username,
                        'avatar' => $comment->user->avatar,
                        'avatar_url' => $comment->user->avatar_url
                    ]);
                }
                
                // Добавляем avatar_url к пользователям в ответах
                if ($comment->replies) {
                    $comment->replies->transform(function ($reply) use ($isAuthenticated, $currentUserId) {
                        $reply->is_liked = false;
                        $reply->can_reply = $isAuthenticated;
                        $reply->can_edit = $isAuthenticated && $reply->user_id === $currentUserId;
                        $reply->can_delete = $isAuthenticated && $reply->user_id === $currentUserId;
                        
                        if ($reply->user) {
                            $reply->user->avatar_url = $reply->user->getAvatarUrl();
                            \Log::info('Reply user avatar', [
                                'user_id' => $reply->user->id,
                                'username' => $reply->user->username,
                                'avatar' => $reply->user->avatar,
                                'avatar_url' => $reply->user->avatar_url
                            ]);
                        }
                        
                        return $reply;
                    });
                }
                
                return $comment;
            });

            $commentsArray = $comments->map(function ($comment) {
                $commentArr = $comment->toArray();
                if ($comment->user) {
                    $commentArr['user']['avatar_url'] = $comment->user->getAvatarUrl();
                }
                if (!empty($comment->replies)) {
                    $commentArr['replies'] = $comment->replies->map(function ($reply) {
                        $replyArr = $reply->toArray();
                        if ($reply->user) {
                            $replyArr['user']['avatar_url'] = $reply->user->getAvatarUrl();
                        }
                        return $replyArr;
                    })->toArray();
                }
                return $commentArr;
            })->toArray();

            return response()->json([
                'success' => true,
                'data' => [
                    'comments' => $commentsArray,
                    'total_comments' => $comments->count(),
                    'pagination' => [
                        'current_page' => 1,
                        'last_page' => 1,
                        'per_page' => 10,
                        'total' => $comments->count(),
                    ]
                ]
            ]);
        } catch (\Exception $e) {
            \Log::error('News comments index error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString(),
                'news_id' => $newsId
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Ошибка при загрузке комментариев: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Создать комментарий
     */
    public function store(Request $request, $newsId): JsonResponse
    {
        try {
            \Log::info('Creating comment', [
                'news_id' => $newsId,
                'request_data' => $request->all()
            ]);

            $request->validate([
                'content' => 'required|string|max:2000',
                'parent_id' => 'nullable|exists:news_comments,id',
            ]);

            $news = News::findOrFail($newsId);
            $user = auth('sanctum')->user();

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'Необходима авторизация'
                ], 401);
            }

            \Log::info('User authenticated', ['user_id' => $user->id]);

            $comment = NewsComment::create([
                'news_id' => $news->id,
                'user_id' => $user->id,
                'parent_id' => $request->parent_id,
                'content' => $request->content,
                'status' => 'approved',
            ]);

            \Log::info('Comment created', [
                'comment_id' => $comment->id,
                'content' => $comment->content,
                'status' => $comment->status
            ]);

            // Загружаем связи
            $comment->load(['user' => function($query) {
                $query->select('id', 'username', 'name', 'avatar');
            }]);

            // Добавляем avatar_url к пользователю (как в getPredictors)
            if ($comment->user) {
                $comment->user->avatar_url = $comment->user->getAvatarUrl();
                \Log::info('Store comment user avatar', [
                    'user_id' => $comment->user->id,
                    'username' => $comment->user->username,
                    'avatar' => $comment->user->avatar,
                    'avatar_url' => $comment->user->avatar_url
                ]);
            }

            // Добавляем информацию о лайках
            $comment->is_liked = false;
            $comment->can_reply = true;

            return response()->json([
                'success' => true,
                'data' => [
                    'comment' => $comment->toArray()
                ],
                'message' => 'Комментарий добавлен'
            ], 201);
        } catch (\Exception $e) {
            \Log::error('News comment store error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString(),
                'news_id' => $newsId,
                'request_data' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Ошибка при добавлении комментария',
                'debug' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Обновить комментарий
     */
    public function update(Request $request, $commentId): JsonResponse
    {
        $request->validate([
            'content' => 'required|string|max:2000',
        ]);

        $comment = NewsComment::findOrFail($commentId);
        $user = auth('sanctum')->user();

        if (!$user || $comment->user_id !== $user->id) {
            return response()->json([
                'success' => false,
                'message' => 'Нет прав для редактирования этого комментария'
            ], 403);
        }

        // Извлекаем упомянутых пользователей
        // $mentionedUsers = $this->extractMentionedUsers($request->content);

        $comment->update([
            'content' => $request->content,
            'is_edited' => true,
            'edited_at' => now(),
        ]);

        $comment->load(['user', 'parent']);

        return response()->json([
            'success' => true,
            'data' => [
                'comment' => $comment
            ],
            'message' => 'Комментарий обновлен'
        ]);
    }

    /**
     * Удалить комментарий
     */
    public function destroy($commentId): JsonResponse
    {
        $comment = NewsComment::findOrFail($commentId);
        $user = auth('sanctum')->user();

        if (!$user || $comment->user_id !== $user->id) {
            return response()->json([
                'success' => false,
                'message' => 'Нет прав для удаления этого комментария'
            ], 403);
        }

        $news = $comment->news;
        $parentComment = $comment->parent;

        $comment->delete();

        // Обновляем счетчики
        $this->updateCommentsCount($news);
        if ($parentComment) {
            $this->updateRepliesCount($parentComment);
        }

        return response()->json([
            'success' => true,
            'message' => 'Комментарий удален'
        ]);
    }

    /**
     * Лайк/дизлайк комментария
     */
    public function like(Request $request, $commentId): JsonResponse
    {
        $comment = NewsComment::findOrFail($commentId);
        $user = auth('sanctum')->user();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Необходима авторизация'
            ], 401);
        }

        $type = $request->get('type', 'like');
        
        // Удаляем существующий лайк/дизлайк
        $comment->likes()->where('user_id', $user->id)->delete();

        // Создаем новый лайк/дизлайк
        $comment->likes()->create([
            'user_id' => $user->id,
            'type' => $type
        ]);

        // Обновляем счетчики
        $this->updateCommentLikesCount($comment);

        return response()->json([
            'success' => true,
            'data' => [
                'likes_count' => $comment->fresh()->likes_count,
                'is_liked' => $type === 'like'
            ]
        ]);
    }

    /**
     * Убрать лайк/дизлайк комментария
     */
    public function unlike($commentId): JsonResponse
    {
        $comment = NewsComment::findOrFail($commentId);
        $user = auth('sanctum')->user();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Необходима авторизация'
            ], 401);
        }

        // Удаляем лайк/дизлайк
        $comment->likes()->where('user_id', $user->id)->delete();

        // Обновляем счетчики
        $this->updateCommentLikesCount($comment);

        return response()->json([
            'success' => true,
            'data' => [
                'likes_count' => $comment->fresh()->likes_count,
                'is_liked' => false
            ]
        ]);
    }

    /**
     * Извлечь упомянутых пользователей из текста
     */
    private function extractMentionedUsers(string $content): array
    {
        preg_match_all('/@(\w+)/', $content, $matches);
        
        if (empty($matches[1])) {
            return [];
        }

        $usernames = $matches[1];
        $users = \App\Models\User::whereIn('username', $usernames)->pluck('id')->toArray();
        
        return $users;
    }

    /**
     * Обновить счетчик комментариев в новости
     */
    private function updateCommentsCount(News $news): void
    {
        $commentsCount = $news->allComments()->approved()->count();
        $news->update(['comments_count' => $commentsCount]);
    }

    /**
     * Обновить счетчик ответов в комментарии
     */
    private function updateRepliesCount(NewsComment $comment): void
    {
        $repliesCount = $comment->replies()->approved()->count();
        $comment->update(['replies_count' => $repliesCount]);
    }

    /**
     * Обновить счетчики лайков комментария
     */
    private function updateCommentLikesCount(NewsComment $comment): void
    {
        $likesCount = $comment->likes()->where('type', 'like')->count();
        $dislikesCount = $comment->likes()->where('type', 'dislike')->count();
        
        $comment->update([
            'likes_count' => $likesCount - $dislikesCount
        ]);
    }
}
