<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\News;
use App\Models\NewsCategory;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class NewsController extends Controller
{
    /**
     * Получить список новостей с фильтрацией
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $query = News::with(['category', 'author'])
                ->published()
                ->latest();

            // Фильтрация по категории
            if ($request->has('category_id')) {
                $query->byCategory($request->category_id);
            }

            // Фильтрация по категории по slug
            if ($request->has('category_slug')) {
                $category = NewsCategory::where('slug', $request->category_slug)->first();
                if ($category) {
                    $query->byCategory($category->id);
                }
            }

            // Поиск
            if ($request->has('search') && !empty($request->search)) {
                $query->search($request->search);
            }

            // Фильтрация по тегам
            if ($request->has('tags') && !empty($request->tags)) {
                $tags = explode(',', $request->tags);
                $query->where(function ($q) use ($tags) {
                    foreach ($tags as $tag) {
                        $q->orWhereJsonContains('tags', trim($tag));
                    }
                });
            }

            // Сортировка
            $sort = $request->get('sort', 'latest');
            switch ($sort) {
                case 'popular':
                    $query->popular();
                    break;
                case 'featured':
                    $query->featured();
                    break;
                case 'pinned':
                    $query->pinned();
                    break;
                default:
                    $query->latest();
            }

            // Пагинация
            $perPage = $request->get('per_page', 12);
            $news = $query->paginate($perPage);

            // Добавляем дополнительную информацию
            $news->getCollection()->transform(function ($item) {
                $item->reading_time = $item->reading_time;
                $item->short_excerpt = $item->short_excerpt;
                $item->url = $item->url;
                
                // Добавляем avatar_url к автору
                if ($item->author) {
                    $item->author->avatar_url = $item->author->getAvatarUrl();
                }
                
                return $item;
            });

            return response()->json([
                'success' => true,
                'data' => [
                    'news' => $news->items(),
                    'pagination' => [
                        'current_page' => $news->currentPage(),
                        'last_page' => $news->lastPage(),
                        'per_page' => $news->perPage(),
                        'total' => $news->total(),
                        'from' => $news->firstItem(),
                        'to' => $news->lastItem(),
                    ]
                ]
            ]);
        } catch (\Exception $e) {
            \Log::error('News index error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Ошибка при загрузке новостей',
                'debug' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Получить конкретную новость
     */
    public function show(Request $request, $slug): JsonResponse
    {
        try {
            $news = News::with(['category', 'author'])
                ->where('slug', $slug)
                ->published()
                ->first();

            if (!$news) {
                return response()->json([
                    'success' => false,
                    'message' => 'Новость не найдена'
                ], 404);
            }

            // Увеличиваем счетчик просмотров
            $this->incrementViews($news, $request);

            // Добавляем дополнительную информацию
            $news->reading_time = $news->reading_time;
            $news->url = $news->url;
            
            // Добавляем avatar_url к автору
            if ($news->author) {
                $news->author->avatar_url = $news->author->getAvatarUrl();
            }

            // Проверяем, лайкнул ли текущий пользователь
            if (auth('sanctum')->check()) {
                $news->is_liked = false; // Временно отключаем
                $news->is_disliked = false; // Временно отключаем
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'news' => $news
                ]
            ]);
        } catch (\Exception $e) {
            \Log::error('News show error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString(),
                'slug' => $slug
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Ошибка при загрузке новости',
                'debug' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Получить категории новостей
     */
    public function categories(): JsonResponse
    {
        try {
            $categories = NewsCategory::active()
                ->ordered()
                ->withCount('publishedNews')
                ->get();

            return response()->json([
                'success' => true,
                'data' => [
                    'categories' => $categories
                ]
            ]);
        } catch (\Exception $e) {
            \Log::error('News categories error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Ошибка при загрузке категорий',
                'debug' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Получить конкретную категорию по slug
     */
    public function showCategory($slug): JsonResponse
    {
        try {
            $category = NewsCategory::where('slug', $slug)
                ->active()
                ->withCount('publishedNews')
                ->first();

            if (!$category) {
                return response()->json([
                    'success' => false,
                    'message' => 'Категория не найдена'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'category' => $category
                ]
            ]);
        } catch (\Exception $e) {
            \Log::error('News category show error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString(),
                'slug' => $slug
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Ошибка при загрузке категории',
                'debug' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Получить популярные новости
     */
    public function popular(Request $request): JsonResponse
    {
        $limit = $request->get('limit', 5);
        
        $news = News::with(['category', 'author'])
            ->published()
            ->popular()
            ->limit($limit)
            ->get();

        $news->transform(function ($item) {
            $item->reading_time = $item->reading_time;
            $item->short_excerpt = $item->short_excerpt;
            $item->url = $item->url;
            
            // Добавляем avatar_url к автору
            if ($item->author) {
                $item->author->avatar_url = $item->author->getAvatarUrl();
            }
            
            return $item;
        });

        return response()->json([
            'success' => true,
            'data' => [
                'news' => $news
            ]
        ]);
    }

    /**
     * Получить выделенные новости
     */
    public function featured(Request $request): JsonResponse
    {
        $limit = $request->get('limit', 3);
        
        $news = News::with(['category', 'author'])
            ->published()
            ->featured()
            ->latest()
            ->limit($limit)
            ->get();

        $news->transform(function ($item) {
            $item->reading_time = $item->reading_time;
            $item->short_excerpt = $item->short_excerpt;
            $item->url = $item->url;
            
            // Добавляем avatar_url к автору
            if ($item->author) {
                $item->author->avatar_url = $item->author->getAvatarUrl();
            }
            
            return $item;
        });

        return response()->json([
            'success' => true,
            'data' => [
                'news' => $news
            ]
        ]);
    }

    /**
     * Лайк/дизлайк новости
     */
    public function like(Request $request, $id): JsonResponse
    {
        $news = News::findOrFail($id);
        $user = auth('sanctum')->user();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Необходима авторизация'
            ], 401);
        }

        $type = $request->get('type', 'like');
        
        // Удаляем существующий лайк/дизлайк
        $news->likes()->where('user_id', $user->id)->delete();

        // Создаем новый лайк/дизлайк
        $news->likes()->create([
            'user_id' => $user->id,
            'type' => $type
        ]);

        // Обновляем счетчики
        $this->updateLikesCount($news);

        return response()->json([
            'success' => true,
            'data' => [
                'likes_count' => $news->fresh()->likes_count,
                'is_liked' => $type === 'like',
                'is_disliked' => $type === 'dislike'
            ]
        ]);
    }

    /**
     * Убрать лайк/дизлайк
     */
    public function unlike(Request $request, $id): JsonResponse
    {
        $news = News::findOrFail($id);
        $user = auth('sanctum')->user();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Необходима авторизация'
            ], 401);
        }

        // Удаляем лайк/дизлайк
        $news->likes()->where('user_id', $user->id)->delete();

        // Обновляем счетчики
        $this->updateLikesCount($news);

        return response()->json([
            'success' => true,
            'data' => [
                'likes_count' => $news->fresh()->likes_count,
                'is_liked' => false,
                'is_disliked' => false
            ]
        ]);
    }

    /**
     * Увеличить счетчик просмотров
     */
    private function incrementViews(News $news, Request $request): void
    {
        $user = auth()->user();
        $ip = $request->ip();
        $sessionId = $request->hasSession() ? $request->session()->getId() : null;

        // Проверяем, не просматривал ли уже пользователь эту новость
        $existingView = $news->views()
            ->where(function ($query) use ($user, $ip, $sessionId) {
                if ($user) {
                    $query->where('user_id', $user->id);
                } else {
                    $query->where('ip_address', $ip);
                    if ($sessionId) {
                        $query->where('session_id', $sessionId);
                    }
                }
            })
            ->where('viewed_at', '>=', now()->subHours(24))
            ->first();

        if (!$existingView) {
            $news->views()->create([
                'user_id' => $user ? $user->id : null,
                'ip_address' => $ip,
                'session_id' => $sessionId,
                'user_agent' => $request->userAgent(),
                'viewed_at' => now(),
            ]);

            $news->increment('views_count');
        }
    }

    /**
     * Обновить счетчики лайков
     */
    private function updateLikesCount(News $news): void
    {
        $likesCount = $news->likes()->where('type', 'like')->count();
        $dislikesCount = $news->likes()->where('type', 'dislike')->count();
        
        $news->update([
            'likes_count' => $likesCount - $dislikesCount
        ]);
    }
}
