<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Prediction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class PredictionController extends Controller
{
    /**
     * Display a listing of predictions
     */
    public function index(Request $request)
    {
        $query = Prediction::public()
            ->with('user:id,name,username,avatar,success_rate');

        if ($request->has('sport')) {
            $query->where('sport', $request->sport);
        }

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->has('date_from')) {
            $query->where('match_date', '>=', $request->date_from);
        }

        if ($request->has('date_to')) {
            $query->where('match_date', '<=', $request->date_to);
        }

        $sortBy = $request->get('sort_by', 'match_date');
        $sortOrder = $request->get('sort_order', 'desc');

        switch ($sortBy) {
            case 'odds':
                $query->orderBy('odds', $sortOrder);
                break;
            case 'stake':
                $query->orderBy('stake', $sortOrder);
                break;
            case 'created_at':
                $query->orderBy('created_at', $sortOrder);
                break;
            default:
                $query->orderBy('match_date', 'desc');
        }

        $predictions = $query->paginate($request->get('per_page', 15));

        // Добавляем avatar_url для пользователей
        $predictionsData = $predictions->toArray();
        $predictionsData['data'] = $predictions->getCollection()->map(function ($prediction) {
            $predictionData = $prediction->toArray();
            $predictionData['user']['avatar_url'] = $prediction->user->getAvatarUrl();
            return $predictionData;
        })->toArray();

        return response()->json([
            'success' => true,
            'data' => [
                'predictions' => $predictionsData
            ]
        ]);
    }

    /**
     * Store a newly created prediction
     */
    public function store(Request $request)
    {
        $user = Auth::user();

        $validator = Validator::make($request->all(), [
            'sport' => 'required|string|max:100',
            'league' => 'required|string|max:100',
            'home_team' => 'required|string|max:100',
            'away_team' => 'required|string|max:100',
            'prediction_type' => 'required|string|max:50',
            'prediction_value' => 'required|string|max:100',
            'odds' => 'required|numeric|min:1.01|max:100',
            'stake' => 'required|numeric|min:1',
            'analysis' => 'nullable|string|max:2000',
            'match_date' => 'required|date|after:now',
            'is_public' => 'boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $prediction = Prediction::create([
            'user_id' => $user->id,
            'sport' => $request->sport,
            'league' => $request->league,
            'home_team' => $request->home_team,
            'away_team' => $request->away_team,
            'prediction_type' => $request->prediction_type,
            'prediction_value' => $request->prediction_value,
            'odds' => $request->odds,
            'stake' => $request->stake,
            'analysis' => $request->analysis,
            'match_date' => $request->match_date,
            'is_public' => $request->get('is_public', true),
        ]);

        // Добавляем avatar_url для пользователя
        $predictionData = $prediction->toArray();
        $predictionData['user']['avatar_url'] = $prediction->user->getAvatarUrl();

        return response()->json([
            'success' => true,
            'message' => 'Prediction created successfully',
            'data' => [
                'prediction' => $predictionData
            ]
        ], 201);
    }

    /**
     * Display the specified prediction
     */
    public function show($id)
    {
        $prediction = Prediction::with('user:id,name,username,avatar,success_rate')
            ->findOrFail($id);

        // Добавляем avatar_url для пользователя
        $predictionData = $prediction->toArray();
        $predictionData['user']['avatar_url'] = $prediction->user->getAvatarUrl();

        return response()->json([
            'success' => true,
            'data' => [
                'prediction' => $predictionData
            ]
        ]);
    }

    /**
     * Update the specified prediction
     */
    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $prediction = Prediction::where('user_id', $user->id)->findOrFail($id);

        // Only allow updates if prediction is still pending
        if ($prediction->status !== 'pending') {
            return response()->json([
                'success' => false,
                'message' => 'Cannot update prediction that is not pending'
            ], 400);
        }

        $validator = Validator::make($request->all(), [
            'analysis' => 'nullable|string|max:2000',
            'is_public' => 'boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $prediction->update($request->only(['analysis', 'is_public']));

        // Добавляем avatar_url для пользователя
        $predictionData = $prediction->toArray();
        $predictionData['user']['avatar_url'] = $prediction->user->getAvatarUrl();

        return response()->json([
            'success' => true,
            'message' => 'Prediction updated successfully',
            'data' => [
                'prediction' => $predictionData
            ]
        ]);
    }

    /**
     * Remove the specified prediction
     */
    public function destroy($id)
    {
        $user = Auth::user();
        $prediction = Prediction::where('user_id', $user->id)->findOrFail($id);

        // Only allow deletion if prediction is still pending
        if ($prediction->status !== 'pending') {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete prediction that is not pending'
            ], 400);
        }

        $prediction->delete();

        return response()->json([
            'success' => true,
            'message' => 'Prediction deleted successfully'
        ]);
    }

    /**
     * Get user's predictions
     */
    public function userPredictions(Request $request)
    {
        $user = Auth::user();

        $query = Prediction::where('user_id', $user->id)
            ->with('user:id,name,username,avatar,success_rate');

        // Apply filters
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('sport')) {
            $query->where('sport', $request->sport);
        }

        $predictions = $query->orderBy('match_date', 'desc')
            ->paginate($request->get('per_page', 15));

        // Добавляем avatar_url для пользователей
        $predictionsData = $predictions->toArray();
        $predictionsData['data'] = $predictions->getCollection()->map(function ($prediction) {
            $predictionData = $prediction->toArray();
            $predictionData['user']['avatar_url'] = $prediction->user->getAvatarUrl();
            return $predictionData;
        })->toArray();

        return response()->json([
            'success' => true,
            'data' => [
                'predictions' => $predictionsData
            ]
        ]);
    }

    /**
     * Get top predictors
     */
    public function topPredictors(Request $request)
    {
        $limit = $request->get('limit', 10);

        $topPredictors = \App\Models\User::where('predictions_count', '>', 0)
            ->orderBy('success_rate', 'desc')
            ->orderBy('correct_predictions', 'desc')
            ->limit($limit)
            ->get(['id', 'name', 'username', 'avatar', 'success_rate', 'predictions_count', 'correct_predictions', 'total_points']);

        // Add avatar_url to users
        $topPredictorsData = $topPredictors->map(function ($user) {
            $userData = $user->toArray();
            $userData['avatar_url'] = $user->getAvatarUrl();
            return $userData;
        })->toArray();

        return response()->json([
            'success' => true,
            'data' => [
                'top_predictors' => $topPredictorsData
            ]
        ]);
    }

    /**
     * Get predictions by sport
     */
    public function bySport($sport, Request $request)
    {
        $query = Prediction::public()
            ->where('sport', $sport)
            ->with('user:id,name,username,avatar,success_rate');

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        $predictions = $query->orderBy('match_date', 'desc')
            ->paginate($request->get('per_page', 15));

        // Добавляем avatar_url для пользователей
        $predictions->getCollection()->transform(function ($prediction) {
            if ($prediction->user) {
                $prediction->user->avatar_url = $prediction->user->getAvatarUrl();
            }
            return $prediction;
        });

        return response()->json([
            'success' => true,
            'data' => [
                'sport' => $sport,
                'predictions' => $predictions
            ]
        ]);
    }

    public function like($id)
    {
        $user = auth()->user();
        
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Пользователь не авторизован.'
            ], 401);
        }
        
        $prediction = \App\Models\Prediction::findOrFail($id);

        // Нельзя лайкать свой прогноз
        if ($prediction->user_id === $user->id) {
            return response()->json([
                'success' => false,
                'message' => 'Вы не можете лайкать свой прогноз.'
            ], 403);
        }

        // Сохраняем лайк
        try {
            \App\Models\PredictionLike::create([
                'user_id' => $user->id,
                'prediction_id' => $id
            ]);
            $prediction->increment('likes_count');
        } catch (\Illuminate\Database\QueryException $e) {
            // Если лайк уже существует (нарушение уникальности)
            if ($e->getCode() == 23000) {
                return response()->json([
                    'success' => false,
                    'message' => 'Вы уже лайкали этот прогноз.'
                ], 409);
            }
            throw $e;
        }

        return response()->json([
            'success' => true,
            'message' => 'Лайк добавлен',
            'likes_count' => $prediction->fresh()->likes_count,
            'is_liked' => true
        ]);
    }

    public function unlike($id)
    {
        $user = auth()->user();
        
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Пользователь не авторизован.'
            ], 401);
        }
        
        $prediction = \App\Models\Prediction::findOrFail($id);

        // Проверяем, есть ли лайк от этого пользователя
        $like = \App\Models\PredictionLike::where('user_id', $user->id)
            ->where('prediction_id', $id)
            ->first();

        if (!$like) {
            return response()->json([
                'success' => false,
                'message' => 'Лайк не найден.'
            ], 404);
        }

        // Удаляем лайк
        $like->delete();
        $prediction->decrement('likes_count');

        return response()->json([
            'success' => true,
            'message' => 'Лайк убран',
            'likes_count' => $prediction->fresh()->likes_count,
            'is_liked' => false
        ]);
    }
}
