<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\PromoCode;
use App\Models\PromoCodeUsage;
use App\Models\UserPoints;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class PromoCodeController extends Controller
{
    /**
     * Display a listing of promo codes
     */
    public function index(Request $request)
    {
        $query = PromoCode::active()
            ->valid()
            ->available()
            ->with('bookmaker');

        // Apply filters
        if ($request->has('bookmaker_id')) {
            $query->where('bookmaker_id', $request->bookmaker_id);
        }

        if ($request->has('featured') && $request->featured) {
            $query->featured();
        }

        if ($request->has('bonus_type')) {
            $query->where('bonus_type', $request->bonus_type);
        }

        if ($request->has('min_bonus')) {
            $query->where('bonus_amount', '>=', $request->min_bonus);
        }

        if ($request->has('currency')) {
            $query->where('bonus_currency', $request->currency);
        }

        // Apply sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');

        switch ($sortBy) {
            case 'bonus_amount':
                $query->orderBy('bonus_amount', $sortOrder);
                break;
            case 'valid_from':
                $query->orderBy('valid_from', $sortOrder);
                break;
            case 'current_uses':
                $query->orderBy('current_uses', $sortOrder);
                break;
            default:
                $query->orderBy('created_at', 'desc');
        }

        $promoCodes = $query->paginate($request->get('per_page', 12));

        return response()->json([
            'success' => true,
            'data' => [
                'promo_codes' => $promoCodes
            ]
        ]);
    }

    /**
     * Display the specified promo code
     */
    public function show($id)
    {
        $promoCode = PromoCode::active()
            ->valid()
            ->available()
            ->with('bookmaker')
            ->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => [
                'promo_code' => $promoCode
            ]
        ]);
    }

    /**
     * Use a promo code
     */
    public function use(Request $request)
    {
        $user = Auth::user();

        $validator = \Validator::make($request->all(), [
            'promo_code_id' => 'required|exists:promo_codes,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $promoCode = PromoCode::findOrFail($request->promo_code_id);

        // Check if promo code is valid
        if (!$promoCode->isValid()) {
            return response()->json([
                'success' => false,
                'message' => 'Promo code is not valid'
            ], 400);
        }

        // Check if user has already used this promo code
        if (PromoCodeUsage::hasUserUsed($user->id, $promoCode->id)) {
            return response()->json([
                'success' => false,
                'message' => 'You have already used this promo code'
            ], 400);
        }

        // Create usage record
        $usage = PromoCodeUsage::create([
            'user_id' => $user->id,
            'promo_code_id' => $promoCode->id,
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'status' => 'confirmed'
        ]);

        // Increment usage count
        $promoCode->incrementUsage();

        // Add points if reward is set
        if ($promoCode->points_reward > 0) {
            UserPoints::addPoints(
                $user->id,
                $promoCode->points_reward,
                'promo_used',
                "Used promo code: {$promoCode->title}",
                $promoCode
            );
        }

        return response()->json([
            'success' => true,
            'message' => 'Promo code used successfully',
            'data' => [
                'promo_code' => $promoCode,
                'usage' => $usage,
                'points_earned' => $promoCode->points_reward
            ]
        ]);
    }

    /**
     * Get featured promo codes
     */
    public function featured()
    {
        $promoCodes = PromoCode::active()
            ->valid()
            ->available()
            ->featured()
            ->with('bookmaker')
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'promo_codes' => $promoCodes
            ]
        ]);
    }

    /**
     * Search promo codes
     */
    public function search(Request $request)
    {
        $query = $request->get('q');

        if (!$query || strlen($query) < 2) {
            return response()->json([
                'success' => false,
                'message' => 'Search query must be at least 2 characters long'
            ], 400);
        }

        $promoCodes = PromoCode::active()
            ->valid()
            ->available()
            ->with('bookmaker')
            ->where(function ($q) use ($query) {
                $q->where('title', 'like', "%{$query}%")
                  ->orWhere('description', 'like', "%{$query}%")
                  ->orWhere('code', 'like', "%{$query}%");
            })
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 10));

        return response()->json([
            'success' => true,
            'data' => [
                'query' => $query,
                'promo_codes' => $promoCodes
            ]
        ]);
    }

    /**
     * Get user's promo code usage history
     */
    public function userHistory(Request $request)
    {
        $user = Auth::user();

        $usages = PromoCodeUsage::where('user_id', $user->id)
            ->with(['promoCode.bookmaker'])
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 10));

        return response()->json([
            'success' => true,
            'data' => [
                'usages' => $usages
            ]
        ]);
    }

    /**
     * Get promo codes statistics
     */
    public function stats()
    {
        $total = PromoCode::active()->valid()->available()->count();
        $active = PromoCode::active()->valid()->available()->where('is_active', true)->count();
        $bookmakers = PromoCode::active()->valid()->available()->distinct('bookmaker_id')->count('bookmaker_id');

        return response()->json([
            'success' => true,
            'data' => [
                'stats' => [
                    'total' => $total,
                    'active' => $active,
                    'bookmakers' => $bookmakers
                ]
            ]
        ]);
    }

    /**
     * Get user's used promo codes
     */
    public function getUserUsedPromoCodes(Request $request)
    {
        $user = Auth::user();

        $usedPromoCodes = PromoCodeUsage::where('user_id', $user->id)
            ->with(['promoCode.bookmaker'])
            ->pluck('promo_code_id')
            ->toArray();

        return response()->json([
            'success' => true,
            'data' => [
                'used_promo_codes' => $usedPromoCodes
            ]
        ]);
    }
}
