<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Story;
use App\Models\StoryView;
use App\Models\StoryLike;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class StoryController extends Controller
{
    /**
     * Get all public stories for homepage.
     */
    public function index(): JsonResponse
    {
        $stories = Story::with(['user:id,username,name,total_points'])
            ->public()
            ->notExpired()
            ->byQualifiedUsers(3000)
            ->orderBy('created_at', 'desc')
            ->get()
            ->groupBy('user_id');

        // Добавляем avatar_url вручную
        foreach ($stories as $userStories) {
            foreach ($userStories as $story) {
                if ($story->user) {
                    $story->user->avatar_url = $story->user->getAvatarUrl();
                }
                
                // Если пользователь не авторизован, устанавливаем значения по умолчанию
                if (!Auth::check()) {
                    $story->is_viewed_by_current_user = false;
                    $story->is_liked_by_current_user = false;
                }
            }
        }

        return response()->json([
            'success' => true,
            'data' => [
                'stories' => $stories
            ]
        ]);
    }

    /**
     * Get user's own stories.
     */
    public function myStories(): JsonResponse
    {
        $stories = Auth::user()->stories()
            ->notExpired()
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'stories' => $stories
            ]
        ]);
    }

    /**
     * Create a new story.
     */
    public function store(Request $request): JsonResponse
    {
        // Проверяем, что у пользователя достаточно баллов
        if (Auth::user()->total_points < 3000) {
            return response()->json([
                'success' => false,
                'message' => 'Для создания сторисов необходимо минимум 3000 баллов'
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'content' => 'nullable|string|max:500',
            'media' => 'nullable|file|mimes:jpg,jpeg,png,gif,mp4,mov|max:10240', // 10MB max
            'type' => 'required|in:text,image,video',
            'is_public' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Ошибка валидации',
                'errors' => $validator->errors()
            ], 422);
        }

        $storyData = [
            'user_id' => Auth::id(),
            'content' => $request->content,
            'type' => $request->type,
            'is_public' => $request->boolean('is_public', true)
        ];

        // Загружаем медиа файл если есть
        if ($request->hasFile('media')) {
            $path = $request->file('media')->store('stories', 'public');
            $storyData['media_url'] = Storage::url($path);
        }

        $story = Story::create($storyData);

        // Загружаем пользователя без avatar_url, добавляем avatar_url вручную
        $story->load('user:id,username,name,total_points');
        if ($story->user) {
            $story->user->avatar_url = $story->user->getAvatarUrl();
        }

        return response()->json([
            'success' => true,
            'message' => 'Сторис успешно создан',
            'data' => [
                'story' => $story
            ]
        ], 201);
    }

    /**
     * Mark story as viewed.
     */
    public function markAsViewed(Story $story): JsonResponse
    {
        // Проверяем, что сторис не просмотрен уже
        if (!$story->is_viewed_by_current_user) {
            StoryView::create([
                'story_id' => $story->id,
                'user_id' => Auth::id()
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Сторис отмечен как просмотренный'
        ]);
    }

    /**
     * Toggle like on story.
     */
    public function toggleLike(Story $story): JsonResponse
    {
        $existingLike = StoryLike::where('story_id', $story->id)
            ->where('user_id', Auth::id())
            ->first();

        if ($existingLike) {
            $existingLike->delete();
            $message = 'Лайк убран';
        } else {
            StoryLike::create([
                'story_id' => $story->id,
                'user_id' => Auth::id()
            ]);
            $message = 'Сторис лайкнут';
        }

        return response()->json([
            'success' => true,
            'message' => $message,
            'data' => [
                'is_liked' => !$existingLike,
                'likes_count' => $story->fresh()->likes_count
            ]
        ]);
    }

    /**
     * Delete story.
     */
    public function destroy(Story $story): JsonResponse
    {
        // Проверяем, что пользователь может удалить свой сторис
        if ($story->user_id !== Auth::id()) {
            return response()->json([
                'success' => false,
                'message' => 'Вы можете удалить только свои сторисы'
            ], 403);
        }

        // Удаляем медиа файл если есть
        if ($story->media_url) {
            $path = str_replace('/storage/', '', $story->media_url);
            Storage::disk('public')->delete($path);
        }

        $story->delete();

        return response()->json([
            'success' => true,
            'message' => 'Сторис успешно удален'
        ]);
    }

    /**
     * Get story statistics.
     */
    public function statistics(): JsonResponse
    {
        $user = Auth::user();
        
        $stats = [
            'total_stories' => $user->stories()->count(),
            'active_stories' => $user->stories()->notExpired()->count(),
            'total_views' => $user->stories()->withCount('views')->get()->sum('views_count'),
            'total_likes' => $user->stories()->withCount('likes')->get()->sum('likes_count'),
            'can_create' => $user->total_points >= 3000
        ];

        return response()->json([
            'success' => true,
            'data' => [
                'statistics' => $stats
            ]
        ]);
    }
}
