<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Bonus extends Model
{
    use HasFactory;

    protected $fillable = [
        'bookmaker_id',
        'title',
        'description',
        'bonus_type',
        'bonus_amount',
        'bonus_currency',
        'min_deposit',
        'max_bonus',
        'wagering_requirement',
        'validity_days',
        'is_active',
        'is_featured',
        'points_reward',
    ];

    protected $casts = [
        'bonus_amount' => 'decimal:2',
        'min_deposit' => 'decimal:2',
        'max_bonus' => 'decimal:2',
        'is_active' => 'boolean',
        'is_featured' => 'boolean',
    ];

    /**
     * Get the bookmaker that owns the bonus.
     */
    public function bookmaker(): BelongsTo
    {
        return $this->belongsTo(Bookmaker::class);
    }

    /**
     * Scope a query to only include active bonuses.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to only include featured bonuses.
     */
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    /**
     * Scope a query to only include bonuses by type.
     */
    public function scopeByType($query, $type)
    {
        return $query->where('bonus_type', $type);
    }

    /**
     * Get formatted bonus amount.
     */
    public function getFormattedAmountAttribute(): string
    {
        if ($this->bonus_amount) {
            return number_format($this->bonus_amount, 0, ',', ' ') . ' ' . $this->bonus_currency;
        }
        
        return 'До ' . number_format($this->max_bonus, 0, ',', ' ') . ' ' . $this->bonus_currency;
    }
}
