<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Prediction extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'sport',
        'league',
        'home_team',
        'away_team',
        'prediction_type',
        'prediction_value',
        'odds',
        'stake',
        'analysis',
        'status',
        'profit_loss',
        'match_date',
        'is_public',
        'likes_count',
        'dislikes_count',
    ];

    protected $casts = [
        'odds' => 'decimal:2',
        'stake' => 'decimal:2',
        'profit_loss' => 'decimal:2',
        'match_date' => 'datetime',
        'is_public' => 'boolean',
        'likes_count' => 'integer',
        'dislikes_count' => 'integer',
    ];

    /**
     * Get the user that owns the prediction.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Scope a query to only include public predictions.
     */
    public function scopePublic($query)
    {
        return $query->where('is_public', true);
    }

    /**
     * Scope a query to only include predictions by status.
     */
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope a query to only include predictions by sport.
     */
    public function scopeBySport($query, $sport)
    {
        return $query->where('sport', $sport);
    }

    /**
     * Calculate success rate for a user.
     */
    public static function calculateSuccessRate($userId): float
    {
        $predictions = self::where('user_id', $userId)
                          ->whereIn('status', ['won', 'lost'])
                          ->get();

        if ($predictions->isEmpty()) {
            return 0.0;
        }

        $won = $predictions->where('status', 'won')->count();
        return round(($won / $predictions->count()) * 100, 2);
    }

    /**
     * Update user statistics after prediction result.
     */
    public function updateUserStats(): void
    {
        $user = $this->user;
        
        if ($this->status === 'won') {
            $user->increment('correct_predictions');
            $user->increment('total_profit', $this->profit_loss);
        }
        
        $user->increment('predictions_count');
        $user->success_rate = self::calculateSuccessRate($user->id);
        $user->save();
    }

    public function likes(): HasMany
    {
        return $this->hasMany(\App\Models\PredictionLike::class);
    }
}
