<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;

class UserPoints extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'points',
        'action_type',
        'description',
        'pointable_type',
        'pointable_id',
    ];

    protected $casts = [
        'pointable_id' => 'integer',
    ];

    /**
     * Get the user that owns the points.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the parent pointable model.
     */
    public function pointable(): MorphTo
    {
        return $this->morphTo();
    }

    /**
     * Scope a query to only include points by action type.
     */
    public function scopeByActionType($query, $actionType)
    {
        return $query->where('action_type', $actionType);
    }

    /**
     * Add points to user.
     */
    public static function addPoints($userId, $points, $actionType, $description, $pointable = null): self
    {
        $data = [
            'user_id' => $userId,
            'points' => $points,
            'action_type' => $actionType,
            'description' => $description,
        ];

        // Add pointable fields if provided
        if ($pointable) {
            $data['pointable_type'] = get_class($pointable);
            $data['pointable_id'] = $pointable->id;
        }

        $userPoints = self::create($data);

        // Update user's total and available points
        $user = User::find($userId);
        $user->increment('total_points', $points);
        $user->increment('available_points', $points);

        return $userPoints;
    }
}
