<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\CategoryController;
use App\Http\Controllers\Api\BookmakerController;
use App\Http\Controllers\Api\CasinoController;
use App\Http\Controllers\Api\SlotController;
use App\Http\Controllers\Api\PromoCodeController;
use App\Http\Controllers\Api\PredictionController;
use App\Http\Controllers\Api\UserController;
use App\Http\Controllers\Api\AdminController;
use App\Http\Controllers\Api\StoryController;
use App\Http\Controllers\Api\ParserController;
use App\Http\Controllers\Api\RankController;
use App\Http\Controllers\Api\BlacklistController;
use App\Http\Controllers\Api\NewsController;
use App\Http\Controllers\Api\NewsCommentController;
use App\Http\Controllers\Api\MessengerController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

// ============================================================================
// ПУБЛИЧНЫЕ МАРШРУТЫ (не требуют авторизации)
// ============================================================================
Route::prefix('v1')->middleware('track.activity')->group(function () {
    
    // Auth routes
    Route::post('/register', [AuthController::class, 'register']);
    Route::post('/login', [AuthController::class, 'login']);
    Route::post('/forgot-password', [AuthController::class, 'forgotPassword']);
    Route::post('/reset-password', [AuthController::class, 'resetPassword']);
    Route::post('/verify-reset-token', [AuthController::class, 'verifyResetToken']);
    
    // Google OAuth routes (require sessions)
    Route::middleware('web')->group(function () {
        Route::get('/auth/google', [\App\Http\Controllers\Api\GoogleAuthController::class, 'redirectToGoogle']);
        Route::get('/auth/google/callback', [\App\Http\Controllers\Api\GoogleAuthController::class, 'handleGoogleCallback']);
    });
    
    // Categories
    Route::get('/categories', [CategoryController::class, 'index']);
    Route::get('/categories/{slug}', [CategoryController::class, 'show']);
    Route::get('/categories/{slug}/bookmakers', [CategoryController::class, 'bookmakers']);
    
    // Bookmakers
    Route::get('/bookmakers', [BookmakerController::class, 'index']);
    Route::get('/bookmakers/search', [BookmakerController::class, 'search']);
    Route::get('/bookmakers/featured', [BookmakerController::class, 'featured']);
    Route::get('/bookmakers/top-rated', [BookmakerController::class, 'topRated']);
    Route::get('/bookmakers/{slug}', [BookmakerController::class, 'show']);
    Route::get('/bookmakers/{slug}/promo-codes', [BookmakerController::class, 'promoCodes']);
    Route::get('/bookmakers/{slug}/bonuses', [BookmakerController::class, 'bonuses']);
    
    // Casinos
    Route::get('/casinos', [CasinoController::class, 'index']);
    Route::get('/casinos/search', [CasinoController::class, 'search']);
    Route::get('/casinos/featured', [CasinoController::class, 'featured']);
    Route::get('/casinos/top-rated', [CasinoController::class, 'topRated']);
    Route::get('/casinos/{slug}', [CasinoController::class, 'show']);
    Route::get('/casinos/{casino}/games', [CasinoController::class, 'games']);
    
    // Slots
    Route::get('/slots', [SlotController::class, 'index']);
    Route::get('/slots/featured', [SlotController::class, 'featured']);
    Route::get('/slots/popular', [SlotController::class, 'popular']);
    Route::get('/slots/{slug}', [SlotController::class, 'show']);
    
    // Promo codes
    Route::get('/promo-codes', [PromoCodeController::class, 'index']);
    Route::get('/promo-codes/stats', [PromoCodeController::class, 'stats']);
    Route::get('/promo-codes/search', [PromoCodeController::class, 'search']);
    Route::get('/promo-codes/featured', [PromoCodeController::class, 'featured']);
    Route::get('/promo-codes/{id}', [PromoCodeController::class, 'show']);
    
    // Predictions
    Route::get('/predictions', [PredictionController::class, 'index']);
    Route::get('/predictions/sport/{sport}', [PredictionController::class, 'bySport']);
    Route::get('/predictions/top-predictors', [PredictionController::class, 'topPredictors']);
    Route::get('/predictions/{id}', [PredictionController::class, 'show']);
    
    // User profiles (public)
    Route::get('/users/{username}', [UserController::class, 'showByUsername']);
    Route::get('/users-online', [UserController::class, 'getOnlineUsers']);
    Route::get('/predictors', [UserController::class, 'getPredictors']);
    Route::get('/leaderboard', [UserController::class, 'leaderboard']);
    
    // Ranks (public)
    Route::get('/ranks', [RankController::class, 'index']);
    Route::get('/ranks/{slug}', [RankController::class, 'show']);
    Route::get('/ranks/by-points', [RankController::class, 'getByPoints']);
    
    // Public stories (for homepage)
    Route::get('/stories', [StoryController::class, 'index']);
    
    // News routes (public)
    Route::get('/news', [NewsController::class, 'index']);
    Route::get('/news/categories', [NewsController::class, 'categories']);
    Route::get('/news/categories/{slug}', [NewsController::class, 'showCategory']);
    Route::get('/news/popular', [NewsController::class, 'popular']);
    Route::get('/news/featured', [NewsController::class, 'featured']);
    Route::get('/news/{slug}', [NewsController::class, 'show']);
    
    // News comments routes (public - только просмотр)
    Route::get('/news/{newsId}/comments', [NewsCommentController::class, 'index']);
    
    // Public statistics (базовая статистика для всех)
    Route::get('/statistics', [AdminController::class, 'publicStats']);
});

// ============================================================================
// ЗАЩИЩЕННЫЕ МАРШРУТЫ (требуют авторизации)
// ============================================================================
Route::prefix('v1')->middleware(['api.auth:sanctum', 'track.activity'])->group(function () {
    
    // User routes
    Route::get('/user/profile', [UserController::class, 'profile']);
    Route::put('/user/profile', [UserController::class, 'updateProfile']);
    Route::post('/user/change-password', [UserController::class, 'changePassword']);
    Route::delete('/user/profile', [UserController::class, 'deleteAccount']);
    Route::post('/user/avatar', [UserController::class, 'uploadAvatar']);
    Route::delete('/user/avatar', [UserController::class, 'removeAvatar']);
    Route::get('/user/predictions', [UserController::class, 'predictions']);
    Route::get('/user/points-history', [UserController::class, 'pointsHistory']);
    Route::get('/user/referrals', [UserController::class, 'referrals']);
    Route::get('/user/follows', [UserController::class, 'follows']);
    Route::get('/user/statistics', [UserController::class, 'statistics']);
    Route::get('/user/stats', [UserController::class, 'statistics']);
    Route::get('/user/promo-codes', [UserController::class, 'promoCodeHistory']);
    
    // User rank progress
    Route::get('/user/rank-progress', [UserController::class, 'rankProgress']);
    
    // Prediction routes
    Route::post('/predictions', [PredictionController::class, 'store']);
    Route::put('/predictions/{id}', [PredictionController::class, 'update']);
    Route::delete('/predictions/{id}', [PredictionController::class, 'destroy']);
    Route::post('/predictions/{id}/like', [PredictionController::class, 'like']);
    Route::delete('/predictions/{id}/like', [PredictionController::class, 'unlike']);
    
    // Promo code routes
    Route::post('/promo-codes/use', [PromoCodeController::class, 'use']);
    Route::get('/promo-codes/user/used', [PromoCodeController::class, 'getUserUsedPromoCodes']);
    
    // Follow routes
    Route::post('/user/follow/{userId}', [UserController::class, 'followUser']);
    Route::delete('/user/follow/{userId}', [UserController::class, 'unfollowUser']);
    Route::get('/user/following', [UserController::class, 'following']);
    Route::get('/user/followers', [UserController::class, 'followers']);
    Route::get('/user/follow-status/{user}', [UserController::class, 'getFollowStatus']);
    Route::post('/user/follow-statuses', [UserController::class, 'getFollowStatuses']);
    
    // Blacklist routes
    Route::get('/user/blacklist', [BlacklistController::class, 'index']);
    Route::post('/user/blacklist/block', [BlacklistController::class, 'block']);
    Route::post('/user/blacklist/unblock', [BlacklistController::class, 'unblock']);
    Route::post('/user/blacklist/check', [BlacklistController::class, 'check']);
    Route::get('/user/blacklist/stats', [BlacklistController::class, 'stats']);
    
    // Stories routes (protected)
    Route::get('/stories/my', [StoryController::class, 'myStories']);
    Route::post('/stories', [StoryController::class, 'store']);
    Route::post('/stories/{story}/view', [StoryController::class, 'markAsViewed']);
    Route::post('/stories/{story}/like', [StoryController::class, 'toggleLike']);
    Route::delete('/stories/{story}', [StoryController::class, 'destroy']);
    Route::get('/stories/statistics', [StoryController::class, 'statistics']);
    
    // News routes (protected)
    Route::post('/news/{newsId}/like', [NewsController::class, 'like']);
    Route::delete('/news/{newsId}/like', [NewsController::class, 'unlike']);
    
    // News comments routes (protected - только создание/редактирование/удаление)
    Route::post('/news/{newsId}/comments', [NewsCommentController::class, 'store']);
    Route::put('/comments/{commentId}', [NewsCommentController::class, 'update']);
    Route::delete('/comments/{commentId}', [NewsCommentController::class, 'destroy']);
    Route::post('/comments/{commentId}/like', [NewsCommentController::class, 'like']);
    Route::delete('/comments/{commentId}/like', [NewsCommentController::class, 'unlike']);

    // ===================== MESSENGER ROUTES =====================
    Route::prefix('messenger')->group(function () {
        Route::get('/conversations', [MessengerController::class, 'conversations']);
        Route::get('/conversations/{conversationId}/messages', [MessengerController::class, 'messages']);
        Route::post('/conversations/{conversationId}/messages', [MessengerController::class, 'sendMessage']);
        Route::post('/conversations/start/{userId}', [MessengerController::class, 'startConversation']);
        Route::post('/conversations/{conversationId}/mark-read', [MessengerController::class, 'markAsRead']);
        Route::delete('/conversations/{conversationId}', [MessengerController::class, 'deleteConversation']);
        // Добавьте другие методы, если нужно
    });
});

// ============================================================================
// АДМИНСКИЕ МАРШРУТЫ (требуют авторизации и права администратора)
// ============================================================================
Route::prefix('v1')->middleware(['api.auth:sanctum', 'admin', 'track.activity'])->group(function () {
    
    // Admin dashboard
    Route::get('/admin/dashboard', [AdminController::class, 'dashboard']);
    Route::get('/admin/statistics', [AdminController::class, 'systemStats']);
    
    // User management
    Route::get('/admin/users', [AdminController::class, 'users']);
    Route::put('/admin/users/{userId}/role', [AdminController::class, 'updateUserRole']);
    Route::put('/admin/users/{userId}/toggle-status', [AdminController::class, 'toggleUserStatus']);
    
    // Bookmaker management
    Route::get('/admin/bookmakers', [AdminController::class, 'bookmakers']);
    Route::post('/admin/bookmakers', [AdminController::class, 'storeBookmaker']);
    Route::put('/admin/bookmakers/{bookmakerId}', [AdminController::class, 'updateBookmaker']);
    Route::put('/admin/bookmakers/{bookmakerId}/toggle-status', [AdminController::class, 'toggleBookmakerStatus']);
    
    // Casino management
    Route::get('/admin/casinos', [AdminController::class, 'casinos']);
    Route::post('/admin/casinos', [AdminController::class, 'storeCasino']);
    Route::put('/admin/casinos/{casinoId}', [AdminController::class, 'updateCasino']);
    Route::put('/admin/casinos/{casinoId}/toggle-status', [AdminController::class, 'toggleCasinoStatus']);
    Route::post('/admin/casinos/{casinoId}/upload-logo', [AdminController::class, 'uploadCasinoLogo']);
    Route::delete('/admin/casinos/{casinoId}', [AdminController::class, 'deleteCasino']);
    Route::post('/admin/casinos/{casinoId}/duplicate', [AdminController::class, 'duplicateCasino']);
    
    // Promo code management
    Route::get('/admin/promo-codes', [AdminController::class, 'promoCodes']);
    Route::post('/admin/promo-codes', [AdminController::class, 'storePromoCode']);
    Route::put('/admin/promo-codes/{promoCodeId}', [AdminController::class, 'updatePromoCode']);
    Route::put('/admin/promo-codes/{promoCodeId}/toggle-status', [AdminController::class, 'togglePromoCodeStatus']);
    
    // Prediction management
    Route::get('/admin/predictions', [AdminController::class, 'predictions']);
    Route::put('/admin/predictions/{predictionId}/status', [AdminController::class, 'updatePredictionStatus']);
    
    // Parser routes
    Route::prefix('parser')->group(function () {
        Route::get('/jobs', [ParserController::class, 'index']);
        Route::post('/jobs', [ParserController::class, 'store']);
        Route::get('/jobs/{parserJob}', [ParserController::class, 'show']);
        Route::put('/jobs/{parserJob}', [ParserController::class, 'update']);
        Route::delete('/jobs/{parserJob}', [ParserController::class, 'destroy']);
        Route::post('/jobs/{parserJob}/start', [ParserController::class, 'start']);
        Route::post('/jobs/{parserJob}/stop', [ParserController::class, 'stop']);
        Route::get('/jobs/{parserJob}/progress', [ParserController::class, 'progress']);
        Route::get('/jobs/{parserJob}/posts', [ParserController::class, 'posts']);
        Route::get('/jobs/{parserJob}/export', [ParserController::class, 'export']);
        Route::post('/test-selectors', [ParserController::class, 'testSelectors']);
    });
});

// ============================================================================
// ДОПОЛНИТЕЛЬНЫЕ МАРШРУТЫ
// ============================================================================
Route::middleware('api.auth:sanctum')->get('/user', function (Request $request) {
    return $request->user();
});

// Тестовый эндпоинт для проверки аутентификации
Route::middleware('api.auth:sanctum')->get('/test-auth', function (Request $request) {
    return response()->json([
        'success' => true,
        'message' => 'Authentication working',
        'user_id' => $request->user()->id,
        'user_email' => $request->user()->email,
        'headers' => $request->header(),
        'bearer_token' => $request->bearerToken()
    ]);
}); 